﻿# lascanopy

This tool reads LiDAR from LAS/LAZ/BIN/SHP/QFIT/ASCII, computes
popular forestry metrics, and grids them onto a raster. A very
important parameter is '-step n' that specifies the n x n area
of LiDAR points that are gridded on one raster (or pixel). The
default of step is 20. The output can be either in BIL, ASC, IMG,
TIF, XYZ, FLT, DTM or LAS/LAZ format.
In CSV format you can request the '-centroids' to be added.
In order to shift the raster grid that the points are binned into away from
the default alignment of (0/0) to (5/15) use '-grid_ll 5 15'.

If your input files are plots you can use the '-files_are_plots'
option. Here you can ask for the file '-names' to be added to
the output CSV file. You can also query a list of circular plots
from a text file with each line listing: "center_x center_y radius"
with a command like:

lascanopy64 -i forest\*.laz -loc circles.txt -cov -p 50 95 -o plots.csv

And if you want to override the radius then you can do this by
adding a fix radius with '-loc_radius 2' or '-loc_radius 7.5'
to the command above. It is also possible to use text files
where each line consists of "name center_x center_y radius" as
the list of circular plots. Then add '-names' before '-loc' to
handle this correctly.

You can also query a list of rectangular plots from a text
file with each line listing: "min_x min_y max_x max_y" with a
command like:

lascanopy64 -i forest\*.laz -lor rectangles.txt -dns -gap -b 50 75 -o stands.csv

It is also possible to use text files where each line consists
of "name min_x min_y max_x max_y" as the list of rectangular
plots. Then add '-names' before '-lor' to handle this correctly.

You can also load more general polygonal plots from a shapefile with

lascanopy64 -i forest\*.laz -lop polygons.shp -int_p 25 50 75 -centroids -o results.csv

If the SHP file with plots has a DBF file with  that contain either an
integer number or a string attributes for each plot you can add the name
or the index of the field to the argument as shown below:

-lop polygons.shp plot_ID
-lop polygons.shp 3

The tool can concurrently compute a number height percentiles
('-p 5 10 25 50 75 90'), the '-min', the '-max', the average
'-avg', and the standard deviation '-std' of all heights above
the cutoff that by default is breast height of 1.37. It can be
changed with the option '-height_cutoff 2.0'. Also the skewness
with '-ske', the kurtosis with '-kur', and the average square
height '-qav' can be computed. All these statistical metrics
only consider the points above the height cutoff. You can compute
the number of points that actually are above the cutoff and are
participating in the computation with '-abv' and the total number
of points (including those that are below the cutoff) with '-all'.

With the command '-s_upper 95' you can limit the computation of
these statistics -qav -avg -ske -kur -std to use only the highest
95 percent of the points above the height cutoff.

There is also the concept of height "bincentiles" where '-b 90'
would deliver the percentage or fraction of points between the
height cutoff (aka breast height) and the maximum height. Hence
the results may not be what you want for '-height_cutoff -2'.
To utilize all of the points for the bincentile calculation use
'-height_cutoff 0' and the transform '-clamp_z_below 0.0'. There
is also option '-b_upper 97' that specifies a certain height
percentile to be used instead of the maximum height as "upper"
limit for the bins. This metric is sometimes also referred to
as "deciles". Request multiple bincentiles with '-b 50 75 90'.

The tool can also produce the canopy cover using option '-cov'.
The canopy cover is computed as the number of first returns
above the cover cutoff divided by the number of all first
returns and output as a percentage. Similarly, with the option
'-dns' the canopy density can be produced. The canopy density
is computed as the number of all points above the cover cutoff
divided by the number of all returns. By default this cover 
cutoff is identical to the height cutoff. However, using the
option '-cover_cutoff 5.0' you can set it to a lower or a
higher value. As the output default percentages between 0.0%
and 100.0% are produced. Use option '-fractions' to produce
fractions between 0.000 and 1.000 instead. 

It is possible to compute the inverse of the canopy cover and
canopy density with the '-gap' option which will give you 100%
(or for fractions 1.0) minus canopy cover or canopy density.

In addition, the tool can also concurrently produce several
height count rasters. The option '-c 0.5 2 4 10 50', for example,
would compute four rasters that count the points whose heights
are falling into the intervals: 
[0.5, 2], [2, 4], [4, 10], and [10, 50]. 
In the same manner the option '-d 0.5 2 4 10 50' will
produce a relative height density raster in which the counts are
divided by the total number of points and scaled to a percentage.

The Vertical Complexity Index (VCI) is also implemented and can
be computed for different vertical bin sizes with '-vc 1 2 4' or
'-vci 2.5 5.0'.

The height & intensity metric known as "Height of Median Energy"
or "HOME" can be computed via the switch '-hom'. All points above
the height cutoff are ordered by their elevation. Then the height
is computed at which the sum of intensities of points below and
the sum of intensities of points above is identical.

Metrics also exist for intensities and '-int_min' and '-int_max'
do the obvious, just like '-int_avg', '-int_qav', '-int_std',
'-int_ske' or '-int_kur'. Similarly you can produce intensity
percentiles with '-int_p 25 50 75' or intensity counts as well
as densities using the corresponding '-int_c 0 128 256 1024' or
'-int_d 0 128 256 1024'.

By default the generated raster spans the extend of the header
bounding box. You can use the bounding box of the tile with
'-use_tile_bb' (which only makes sense if the LAS/LAZ file was
generated using lastile) or the original bounding box in case
of a buffered tile with '-use_orig_bb' (which only makes sense
if the input has an on-the-fly buffer aka '-buffered 50'). The
extend can also be defined by setting '-ll min_x min_y' plus
'-ncols 512' and '-nrows 512'.

For the height_cutoff to make sense it is important that the input
is height normalized, meaning that the z coordinate of each point
corresponds to the height above ground and not the elevation of
the point. With 'lasheight -i in.laz -replace_z -o out.laz' you
can height-normalize a ground-classified LiDAR file.

Depending on the height cutoff or the input file it is possible
that generated rasters contain only 'no data' values but not a
single real value. Adding option '-remove_empty_rasters' to the
command line will delete those files from disk.

Let us know which other metrics you would like to see ...


## Examples

    lascanopy64 -i *.las -min -max -avg

for each *.las files and for all height above 1.37 it computes
the minimum, maximum, and average value from all points that fall
into cells of size 20 by 20 and stores the resulting grid in ASC
format using the endings '_min.asc', '_max.asc', '_avg.asc'.

    lascanopy64 -i lidar*.laz -merged -p 20 40 60 80 -step 10 -o dem.bil

merges the points of all files that match the wildcard lidar*.laz
on-the-fly into one file and computes for all heights above 1.37
the 20th, 40th, 60th, and 80th percentile for 10 by 10 grid cells
and stores the resulting rasters in BIL format using the endings
'_p20.bil', '_p40.bil', '_p60.bil', and '_p80.bil'.


lascanopy64 -h  
lascanopy64 -i *.las -max -avg -qav -p 50 95  
lascanopy64 -i *.laz -p 1 5 10 25 50 75 90 95 99  
lascanopy64 -i *.laz -c 2.0 5.0 10.0 20.0 -cov -dns -otif  
lascanopy64 -i *.laz -merged -max -avg -int_p 5 50 95 -o merged.dtm  
lascanopy64 -i *.laz -d 2.0 10.0 20.0 40.0 -cov -height_cutoff 2.0  
lascanopy64 -i *.laz -int_p 25 50 75 -dns -gap -fractions -otif  
lascanopy64 -i *.las -files_are_plots -int_avg -int_std -cov -gap  
lascanopy64 -i *.laz -p 25 50 75 95 -loc list_of_circles.txt  
lascanopy64 -i *.laz -d 2.0 4.0 6.0 8.0 -fractions -lor list_of_rectangles.txt  
lascanopy64 -i *.laz -p 95 -int_p 95 -cov -fractions -lop list_of_polygons.shp  
lascanopy64 -i 2014_07.laz -ll 470000 5550000 -step 10 -ncols 500 -nrows 200 -cov -p 50 95
lascanopy64 -i *.laz -ocsv -ground_class 2 -p_ith_return 3 4 -l_mom -pzabovex 5 -above_mean

## lascanopy arguments for metrics 

-abv                     : report number of points above the cutoff  
-all                     : report total number of points  
-avg                     : for each grid cell compute average  
-b [n]                   : deliver the percentage or fraction of points between the height cutoff (aka breast height) and the maximum height [n]  
-b_upper [n]             : specifies a certain height [n] percentile to be used instead of the maximum height as "upper" limit for the bins  
-c [m] [n] [o] ...       : compute those height rasters [m-n,n-o,...]  
-cov                     : cover is computed as the number of first returns above the cover cutoff divided by the number of all first returns and output as a percentage  
-d [m] [n] [o] ...       : compute a height raster [m-n,n-o,...] in which the counts are divided by the total number of points and scaled to a percentage  
-dns                     : compute the canopy density as the number of all points above the cover cutoff divided by the number of all returns  
-gap                     : compute the inverse of the canopy cover (100% - canopy)  
-gps_max                 : compute GPS max value  
-gps_min                 : compute GPS min value  
-hom                     : compute the height & intensity metric known as "Height of Median Energy"  
-kur                     : compute kurtosis / bulge  
-max                     : for each grid cell keep highest value  
-min                     : for each grid cell keep lowest value  
-p [m] [n] [o] ...       : compute height percentiles [m] [n] [o] ...  
-qav                     : computer the average square height above the height cutoff  
-ske                     : compute height skewness  
-std                     : for each grid cell compute standard deviation  
-vc [m] [n] [o] ...      : compute the Vertical Complexity Index (VCI) with bin sizes width of [m] [n] [o] ...  
-above_mean              : percentage of returns above mean height (Zmean). Reports percentage of points which are higher than the average height.
-pzabovex [x]            : percentage of returns above X. Reports the proportion of all returns above the specified height X.
-p_ith_return [m] [n]... : percentage of i-th returns. Reports what percentage of all returns are specifically the n-th return.
-ground_class [c] [d]... : percentage of returns classified as ground with classification [c]. Reports the proportion of points with the given classification code (e.g., c=2=ground).
-l_mom                   : compute L-moments of the height distribution.  L-L₂ = spread, L₃ = L-skewness, L₄ = L-kurtosis, τ₃ = L-skewness (L₃/L₂), τ₄ = L-kurtosis (L₄/L₂).

-int_avg                 : compute intensity average  
-int_c [m] [n] [o] ...   : produce intensity bands with gap values [m] [n] [o] ...  
-int_d [m] [n] [o] ...   : produce intensity bands with absolute values [m] [n] [o] ...  
-int_kur                 : compute intensity kurtosis  
-int_max                 : compute intensity maximum  
-int_min                 : compute intensity minimum  
-int_p [m] [n] [o] ...   : produce intensity percentiles with percent values [m] [n] [o] ...  
-int_qav                 : compute intensity quadratic average  
-int_ske                 : compute intensity skewness  
-int_std                 : compute intensity standard deviation  

-red_avg                 : compute red color value average  
-red_c [m] [n] [o] ...   : produce red color value bands with gap values [m] [n] [o] ...  
-red_d [m] [n] [o] ...   : produce red color value bands with absolute values [m] [n] [o] ...  
-red_kur                 : compute red color value kurtosis  
-red_max                 : compute red color value maximum  
-red_min                 : compute red color value minimum  
-red_p [m] [n] [o] ...   : produce red color value percentiles with percent values [m] [n] [o] ...  
-red_qav                 : compute red color value quadratic average  
-red_ske                 : compute red color value skewness  
-red_std                 : compute red color value standard deviation  

-green_avg               : compute green color value average  
-green_c [m] [n] [o] ... : produce green color value bands with gap values [m] [n] [o] ...  
-green_d [m] [n] [o] ... : produce green color value bands with absolute values [m] [n] [o] ...  
-green_kur               : compute green color value kurtosis  
-green_max               : compute green color value maximum  
-green_min               : compute green color value minimum  
-green_p [m] [n] [o] ... : produce green color value percentiles with percent values [m] [n] [o] ...  
-green_qav               : compute green color value quadratic average  
-green_ske               : compute green color value skewness  
-green_std               : compute green color value standard deviation  

-blue_avg                : compute blue color value average  
-blue_c [m] [n] [o] ...  : produce blue color value bands with gap values [m] [n] [o] ...  
-blue_d [m] [n] [o] ...  : produce blue color value bands with absolute values [m] [n] [o] ...  
-blue_kur                : compute blue color value kurtosis  
-blue_max                : compute blue color value maximum  
-blue_min                : compute blue color value minimum  
-blue_p [m] [n] [o] ...  : produce blue color value percentiles with percent values [m] [n] [o] ...  
-blue_qav                : compute blue color value quadratic average  
-blue_ske                : compute blue color value skewness  
-blue_std                : compute blue color value standard deviation  

-nir_avg                 : compute near infrared valuel average  
-nir_c [m] [n] [o] ...   : produce near infrared valuel bands with gap values [m] [n] [o] ...  
-nir_d [m] [n] [o] ...   : produce near infrared valuel bands with absolute values [m] [n] [o] ...  
-nir_kur                 : compute near infrared valuel kurtosis  
-nir_max                 : compute near infrared valuel maximum  
-nir_min                 : compute near infrared valuel minimum  
-nir_p [m] [n] [o] ...   : produce near infrared valuel percentiles with percent values [m] [n] [o] ...  
-nir_qav                 : compute near infrared valuel quadratic average  
-nir_ske                 : compute near infrared valuel skewness  
-nir_std                 : compute near infrared valuel standard deviation  

## lascanopy other specific arguments

-centroids                 : add centroids to CSV output  
-copy_attribute_into_z [n] : copy attribute [n] value into z  
-cover_cutoff [n]          : set level for -cov or -dns calculation to [n]{default="-height_cutoff"}  
-elevation_feet            : use feet for elevation  
-feet                      : use feet  
-files_are_plots           : input files are plot names  
-fractions                 : use fraction (0.000..1.000) instead of percentage as output  
-grid_ll [x] [y]           : shift raster grid lower left to [x]/[y]  
-height_cutoff [n]         : set height cutoff to [n] (default=1.37) in meters  
-height_in_attribute [n]   : use height above ground stored in extra attribute [n]{0=first} instead of from user data  
-ilay [n]                  : apply [n] or all LASlayers found in corresponding *.lay file on read  
-ilaydir [n]               : look for corresponding *.lay file in directory [n]  
-ll [x] [y]                : start rastering at these lower left [x] and [y] coordinates  
-loc [fnt]                 : use file [fnt] as list of circular plots "name center_x center_y radius"  
-loc_radius [n]            : set radius to [n]  
-lop [m] [n]               : use input out of a list of points [m] using plot_ID [n]  
-lor [fnt]                 : use file [fnt] as list of rectangular plots as "name min_x min_y max_x max_y"  
-mem [n]                   : use [n] MB of main memory (500-2000; default=1500)  
-names                     : use names of plots in output  
-ncols [n]                 : raster at most [n] columns  
-nodata [n]                : use [n] as the nodata value in the BIL/ASC format  
-nrows [n]                 : raster at most [n] rows  
-remove_empty_raster       : remove empty rasters  
-remove_empty_rasters      : remove raster files containing only 'nodata' values  
-s_upper [n]               : limit the computation of the statistics [-qav,-avg,-ske,-kur,-std] to use only the highest [n] percent of the points above the height cutoff  
-stdout                    : pipe to stdout  
-step [n]                  : raster with stepsize [n] (default=20)  
-use_bb                    : raster area specified by bounding box in LAS header  
-use_orig_bb               : raster tile without buffer added by on-the-fly buffering  
-use_tile_bb               : raster tile without buffer added by lastile  
-week_to_adjusted [n]      : converts time stamps from GPS week [n] to Adjusted Standard GPS  

### Basics
-cores [n]      : process multiple inputs on [n] cores in parallel  
-license        : show license information  
-demo           : use LAStools in demo mode (64bit only)  
-fail           : fail if license expired or invalid (default for 64bit)
-h, -help       : print help output  
-v, -verbose    : verbose output (print extra information)  
-vv             : very verbose output (print even more information)  
-silent         : only output on errors or warnings
-quiet          : no output at all
-force          : continue, even if serious warnings occur  
-errors_ignore  : continue, even if errors occur (if possible). Use with caution!
-print_log_stats: print additional log statistics  
-cpu64          : force 32bit version to start 64 bit in multi core (obsolete)
-gui            : start with files loaded into GUI  
-version        : reports this tool's version number  


## Module arguments

### General
-buffered [n]      : define read or write buffer of size [n]{default=262144}  
-comma_not_point   : use comma instead of point as decimal separator  
-neighbors [n]     : set neighbors filename or wildcard [n]  
-neighbors_lof [n] : set neighbors list of files [fnf]  
-no_data [n]       : use [n] as the nodata value in the BIL / ASC format  
-no_data_alpha     : adds an alpha channel to the GeoTIFF output, designating areas with no data as transparent  
-no_kml            : avoids auto-creation of KML wrapper  
-no_world_file     : avoid world-file for PNG, JPG, TIF and BIL output  
-stored            : use in memory reader  

### Color
-clamp_RGB_to_8bit                  : limit RGB values to 8 bit (otherwise: 16 bit)  
-copy_B_into_NIR                    : copy blue color value into NearInfraRed value  
-copy_B_into_intensity              : copy blue color value to intensity  
-copy_B_into_register [n]           : copy blue color value into register [n]  
-copy_G_into_NIR                    : copy green color value into NearInfraRed value  
-copy_G_into_intensity              : copy green color value to intensity  
-copy_G_into_register [n]           : copy green color value into register [n]  
-copy_NIR_into_intensity            : copy NIR into intensity  
-copy_NIR_into_register [n]         : copy NearInfraRed value into register [n]  
-copy_RGB_into_intensity            : copy weighted RGB value to intensity  
-copy_R_into_NIR                    : copy red color value into NearInfraRed value  
-copy_R_into_intensity              : copy red color value to intensity  
-copy_R_into_register [n]           : copy red color value into register [n]  
-copy_attribute_into_B [n]          : copy attribute [n] value into blue  
-copy_attribute_into_G [n]          : copy attribute [n] value into green  
-copy_attribute_into_NIR [n]        : copy attribute [n] value into NIR (NearInfraRed)  
-copy_attribute_into_R [n]          : copy attribute [n] value into red  
-copy_intensity_into_NIR            : copy intensity into NIR (NearInfraRed) value  
-copy_register_into_B [n]           : copy register [n] into blue color value  
-copy_register_into_G [n]           : copy register [n] into green color value  
-copy_register_into_I [n]           : copy register [n] into NearInfraRed value  
-copy_register_into_NIR [n]         : copy register [n] into NearInfraRed value  
-copy_register_into_R [n]           : copy register [n] into red color value  
-drop_RGB_green [min] [max]         : drop points with green color value between [min] and [max]  
-drop_RGB_red [min] [max]           : drop points with red color value between [min] and [max]  
-force_RGB                          : force the use of the RGB value even if the point format does not support RGB  
-keep_NDVI [min] [max]              : keep NDVI (Normalized Difference Vegetation Index) between [min] [max]  
-keep_NDVI_from_CIR [min] [max]     : keep NDVI (Normalized Difference Vegetation Index) from CIR between [min] [max]  
-keep_NDVI_green_is_NIR [min] [max] : keep NDVI (Normalized Difference Vegetation Index) where green is NIR between [min] [max]  
-keep_NDVI_blue_is_NIR [min] [max]  : keep NDVI (Normalized Difference Vegetation Index) where blue is NIR between [min] [max]  
-keep_NDVI_intensity_is_NIR [min] [max]: keep NDVI (Normalized Difference Vegetation Index) where intensity is NIR between [min] [max]  
-keep_RGB_blue [m] [n]              : keep points with RGB blue color values between [min] [max]  
-keep_RGB_green [min] [max]         : keep points with green color value between [min] and [max]  
-keep_RGB_greenness [m] [n]         : keep points with RGB greenness values between [min] [max]  
-keep_RGB_nir [m] [n]               : keep points with RGB NIR values between [min] [max]  
-keep_RGB_red [min] [max]           : keep points with red color value between [min] and [max]  
-map_attribute_into_RGB [a] [fnm]   : map attribute [a] by table in file [fnm] to RGB values  
-scale_NIR [n]                      : scale NearInfraRed value by factor [n]  
-scale_NIR_down                     : scale NearInfraRed value down by 256  
-scale_NIR_to_16bit                 : scale 8 bit NearInfraRed value to 16 bit  
-scale_NIR_to_8bit                  : scale 16 bit NearInfraRed value downto 8 bit  
-scale_NIR_up                       : scale NearInfraRed value up by 256  
-scale_RGB [r] [g] [b]              : scale RGB values by factors in [r][g][b]  
-scale_RGB_down                     : scale RGB color values down by 256  
-scale_RGB_to_16bit                 : scale 8 bit color values to 16 bit  
-scale_RGB_to_8bit                  : scale 16 bit color values downto 8 bit  
-scale_RGB_up                       : scale RGB values from 8 bit up to 16 bit (multiply with 256)  
-scale_rgb_down                     : divides all RGB values by 256 (to go from 16 bit to 8 bit numbers)  
-scale_rgb_up                       : multiplies all RGB values by 256 (to go from 8 bit to 16 bit numbers)  
-set_NIR [n]                        : set NearInfraRed value to [n]  
-set_RGB [r] [g] [b]                : set color to [r] [g] [b]  
-set_RGB_of_class [c] [r] [g] [b]   : set RGB values of class [c] to [r][g][b] (8 or 16 bit)  
-switch_G_B                         : switch green and blue value  
-switch_RGBI_into_CIR               : set R to NIR; G to R; B to G  
-switch_RGB_intensity_into_CIR      : set R to intensity; G to R; B to G  
-switch_R_B                         : switch red and blue color value  
-switch_R_G                         : switch red and green color value  

### Coordinates
-add_attribute_to_z [n]             : add value of attribute [n] to z value  
-add_scaled_attribute_to_z [m] [n]  : scale attribute [m] value by [n] and add to z value  
-auto_reoffset                      : puts a reasonable offset in the header and translates the points accordingly. Only applicable to LAS/LAZ input files  
-bin_Z_into_point_source [n]        : set point source to z/[n]  
-clamp_raw_z [min] [max]            : limit raw z values to [min] and [max]  
-clamp_z [min] [max]                : limit z values to [min] and [max]  
-clamp_z_above [n]                  : limit z values to maximal [n]  
-clamp_z_below [n]                  : limit z values to minimal [n]  
-classify_z_above_as [m] [n]        : for z value above [m] set class to [n]  
-classify_z_below_as [m] [n]        : for z value below [m] set class to [n]  
-classify_z_between_as [m] [n] [o]  : for z value between [m] and [n] set class to [o]  
-copy_attribute_into_x [n]          : copy attribute [n] value into x  
-copy_attribute_into_y [n]          : copy attribute [n] value into y  
-copy_intensity_into_z              : copy intensity to z value  
-copy_register_into_x [n]           : copy register [n] to x value  
-copy_register_into_y [n]           : copy register [n] to y value  
-copy_register_into_z [n]           : copy register [n] to z value  
-copy_user_data_into_z              : copy user data into z  
-copy_z_into_attribute [n]          : copy z value into attribute [n] value  
-drop_x [m] [n]                     : drop points with x value between [m] and [n]  
-drop_x_above [n]                   : drop points with x value above [n]  
-drop_x_below [n]                   : drop points with x value below [n]  
-drop_xy [x1] [y1] [x2] [y2]        : drop points within the [x1] [y1] [x2] [y2] rectangle  
-drop_xyz [x1] [y1] [z1] [x2] [y2] [z2]: drop points within the given cube dimensions  
-drop_y [m] [n]                     : drop points with y value between [m] and [n]  
-drop_y_above [n]                   : drop points with y value above [n]  
-drop_y_below [n]                   : drop points with y value below [n]  
-drop_z [m] [n]                     : drop points with z value between [m] and [n]  
-drop_z_above [n]                   : drop points with z value above [n]  
-drop_z_below [n]                   : drop points with z value below [n]  
-inside [x1] [y1] [x2] [y2]         : use only points within the [x1] [y1] [x2] [y2] rectangle  
-inside_circle [x] [y] [r]          : keep circle at pos [x] [y] with radius [r]  
-inside_rectangle [x1] [y1] [x2] [y2]: use only points within the [x1] [y1] [x2] [y2] rectangle  
-inside_tile [m] [n] [o]            : use only points inside tile at lower-left [x] [y] with size [s]  
-keep_circle [x] [y] [r]            : keep circle at pos [x] [y] with radius [r]  
-keep_profile [x1] [y1] [x2] [y2] [w]: keep profile with [x1] [y1] [x2] [y2] [w]  
-keep_tile [x] [y] [size]           : keep tile at lower-left [x] [y] with size [s]  
-keep_x [m] [n]                     : keep points with x value between [m] and [n]  
-keep_xy [x1] [y1] [x2] [y2]        : keep points within the [x1] [y1] [x2] [y2] rectangle  
-keep_xyz [x1] [y1] [z1] [x2] [y2] [z2]: keep points within the given cube dimensions  
-keep_y [m] [n]                     : keep points with y value between [m] and [n]  
-keep_z [m] [n]                     : keep points with z value between [m] and [n]  
-keep_z_above [n]                   : keep points with z value above [n]  
-keep_z_below [n]                   : keep points with z value below [n]  
-offset_adjust                      : adjusting the offset based on the results of point operations and transformations
-reoffset [x] [y] [z]               : puts a new offset [x] [y] [z] into the header and translates the points accordingly  
-rescale [x] [y] [z]                : puts a new scale [x] [y] [z] into the header and rescales the points accordingly  
-rescale_xy [x] [y]                 : rescale x y by [x] [y]  
-rescale_z [z]                      : rescale z by [z]  
-rotate_xy [a] [x] [y]              : rotate points by [a] degrees, center at [x] [y]  
-rotate_xz [a] [x] [z]              : rotate points by [a] degrees, center at [x] [z]  
-rotate_yz [a] [y] [z]              : rotate points by [a] degrees, center at [y] [z]  
-scale_x [n]                        : scale x value by [n]  
-scale_xyz [m] [n] [o]              : scale xyz values by [m] [n] [o]  
-scale_y [n]                        : scale y value by [n]  
-scale_z [n]                        : scale z value by [n]  
-switch_x_y                         : exchange x and y value  
-switch_x_z                         : exchange x and z value  
-switch_y_z                         : exchange z and x value  
-transform_affine [k,w,x,y]         : transform by k*cos(w)+k*sin(w)+tx;k*cos(w)+k*sin(w)+ty - angle in second of arc  
-transform_helmert [m] [n] [o]      : do a helmert transformation with 3 or 7 comma separated parameters [n] ...  
-transform_matrix [r11,r12,r13] [r21,r22,r23] [r31,r32,r33] [tr1,tr2,tr3]: transform input using matrix [r11,r12,r13] [r21,r22,r23] [r31,r32,r33] [tr1,tr2,tr3]  
-translate_raw_x [n]                : translate raw x value by [n]  
-translate_raw_xy_at_random [x] [y] : translate raw xy values by random and max offset of [x] [y]  
-translate_raw_xyz [x] [y] [z]      : translate raw coordinates by [x] [y] [z]  
-translate_raw_y [n]                : translate raw y value by [n]  
-translate_raw_z [n]                : translate raw z value by [n]  
-translate_scale_translate_x [m] [n]: calculate x value as (x–[m])*[n]+[m]  
-translate_scale_translate_y [m] [n]: calculate y value as (y–[m])*[n]+[m]  
-translate_scale_translate_z [m] [n]: calculate z value as (z–[m])*[n]+[m]  
-translate_then_scale_x [m] [n]     : translate x value by [m] and scale by [n]  
-translate_then_scale_y [m] [n]     : translate y value by [m] and scale by [n]  
-translate_then_scale_z [m] [n]     : translate z value by [m] and scale by [n]  
-translate_x [n]                    : translate y value by [n]  
-translate_xyz [x] [y] [z]          : translate point coordinates by [x] [y] [z]  
-translate_y [n]                    : translate y value by [n]  
-translate_z [n]                    : translate z value by [n]  

### Simple thinning
-drop_every_nth [n]           : drop every [n]th point  
-keep_every_nth [n]           : keep every [n]th point  
-keep_random_fraction [m] [n] : keep points by random fraction [m]{0-1}, optional seed [n]  
-thin_points_with_time [n]    : thin points with time, [n] = timespacing  
-thin_pulses_with_time [n]    : thin pulses with time, [n] = timespacing  
-thin_with_grid [n]           : thin points by min grid size of [n]  
-thin_with_time [n]           : thin pulses with time, [n] = timespacing  

### Return number
-change_extended_number_of_returns_from_to [m] [n]: change extended number of returns from [m] to [n]  
-change_extended_return_number_from_to [m] [n]: change extended return number from [m] to [n]  
-change_number_of_returns_from_to [m] [n]: change number of returns from [m] to [n]  
-change_return_number_from_to [m] [n]: change return number from [m] to [n]  
-drop_double                        : drop double returns  
-drop_first                         : drop first return  
-drop_first_of_many                 : drop first of many returns  
-drop_last                          : drop last return  
-drop_last_of_many                  : drop last of many returns  
-drop_middle                        : drop middle returns  
-drop_number_of_returns [n]         : drop points with [n] number of returns  
-drop_quadruple                     : drop quadruple returns  
-drop_quintuple                     : drop quintuple returns  
-drop_return [m] [n]...             : drop points with return [m] [n]...  
-drop_return_mask [n]               : drop points with return mask [n]  
-drop_second_last                   : drop points with second last return  
-drop_single                        : drop points with single return  
-drop_triple                        : drop points with triple return  
-first_only                         : use first return only  
-keep_double                        : keep double returns  
-keep_first                         : keep first return  
-keep_first_of_many                 : keep first of many returns  
-keep_last                          : keep last return  
-keep_last_of_many                  : keep last of many returns  
-keep_middle                        : keep mittle returns  
-keep_number_of_returns [n]         : keep points with [n] number of returns  
-keep_quadruple                     : keep quadruple returns  
-keep_quintuple                     : keep quintuple returns  
-keep_return [m] [n]...             : keep points with return [m] [n]...  
-keep_return_mask [n]               : keep points with return mask [n]  
-keep_second_last                   : keep points with second last return  
-keep_single                        : keep points with single return  
-keep_triple                        : keep points with triple return  
-last_only                          : use last return only  
-repair_zero_returns                : sets return counts and number of returns that are zero to one  
-set_extended_number_of_returns [n] : set extended number of returns to [n]  
-set_extended_return_number [n]     : set extended return number to [n]  
-set_number_of_returns [n]          : set number of returns to [n]  
-set_return_number [n]              : set return number to [n]  

### Scanline
-drop_scan_direction [n]       : drop points with scan direction [n]  
-faf                           : input files are flightlines. do ***NOT*** use this for tiled input  
-faf_index [n]                 : set files are flightlines index [n]  
-files_are_flightlines         : input files are flightlines. do ***NOT*** use this for tiled input  
-drop_edge_of_flight_line      : drop points with "Edge of Flight Line" flag set  
-keep_edge_of_flight_line      : keep points with "Edge of Flight Line" flag set  
-keep_scan_direction_change    : keep points with changed scan direction flag  
-set_edge_of_flight_line [0/1] : set "Edge of Flight Line" flag to [0/1]  
-set_scan_direction_flag [0/1] : set scan direction flag to [0/1]  

### Scanner channel
-copy_scanner_channel_into_point_source: copy scanner channel into point_source  
-copy_scanner_channel_into_user_data: copy scanner channel into user data  
-copy_user_data_into_scanner_channel: copy user data into scanner channel  
-drop_scanner_channel [n]           : drop points with scanner channel [n]  
-keep_scanner_channel [n]           : keep points with scanner channel [n]  
-merge_scanner_channel_into_point_source: merge scanner channel to point source  
-set_extended_scanner_channel [n]   : set extended scanner channel to [n]  
-set_scanner_channel [n]            : set scanner channel to [n]  
-split_scanner_channel_from_point_source: split scanner channel from point source and save as extended scanner channel  

### Source ID
-apply_file_source_ID               : copy file source ID to target  
-bin_Z_into_point_source [n]        : set point source to z/[n]  
-bin_abs_scan_angle_into_point_source [n]: set point source to scan_angle/[n]  
-bin_gps_time_into_point_source [n] : set point source to gps/[n]  
-change_point_source_from_to [m] [n]: change point source from [m] to [n]  
-copy_attribute_into_point_source [n]: copy attribute [n] value into point source  
-copy_classification_into_point_source: copy classification to point source  
-copy_point_source_into_register [n]: copy point source into register [n]  
-copy_register_into_point_source [n]: copy register [n] to point source  
-copy_scanner_channel_into_point_source: copy scanner channel into point_source  
-copy_user_data_into_point_source   : copy user data into point source  
-drop_point_source [n]              : drop points with point source [n]  
-drop_point_source_above [n]        : drop points with with point source above [n]  
-drop_point_source_below [n]        : drop points with with point source below [n]  
-drop_point_source_between [m] [n]  : drop points with with point source between [n] and [m]  
-keep_point_source [n]              : keep points with point source [n]  
-keep_point_source_between [m] [n]  : keep points with with point source between [n] and [m]  
-map_point_source [fnm]             : set the point source by map in file [fnm]  
-merge_scanner_channel_into_point_source: merge scanner channel to point source  
-set_point_source [n]               : set point source to [n]  
-split_scanner_channel_from_point_source: split scanner channel from point source and save as extended scanner channel  

### User data
-add_scaled_attribute_to_user_data [m] [n]: scale attribute [m] value by [n] and add to user data  
-change_user_data_from_to [m] [n]   : change user data from [m] to [n]  
-copy_attribute_into_user_data [n]  : copy attribute [n] value into user data field  
-copy_classification_into_user_data : copy classification to user data  
-copy_register_into_user_data [n]   : copy register [n] to user data  
-copy_scanner_channel_into_user_data: copy scanner channel into user data  
-copy_user_data_into_attribute [n]  : copy user data into attribute [n] value  
-copy_user_data_into_classification : copy user data into classification  
-copy_user_data_into_point_source   : copy user data into point source  
-copy_user_data_into_register [n]   : copy user data to register [n]  
-copy_user_data_into_scanner_channel: copy user data into scanner channel  
-copy_user_data_into_z              : copy user data into z  
-drop_user_data [n]                 : drop points with user data value of [n]  
-drop_user_data_above [n]           : drop points with user data value above [n]  
-drop_user_data_below [n]           : drop points with user data value below [n]  
-drop_user_data_between [m] [n]     : drop points with user data between [m] and [n]  
-keep_user_data [n]                 : keep points with user data value of [n]  
-keep_user_data_above [n]           : keep points with user data value above [n]  
-keep_user_data_below [n]           : keep points with user data value below [n]  
-keep_user_data_between [m] [n]     : keep points with user data between [m] and [n]  
-map_user_data [fnm]                : set the user data by map in file [fnm]  
-scale_user_data [n]                : scale user data by [n]  
-set_user_data [n]                  : sets all user_data fields to [n]  

### Classification
-change_class_from_to [m] [n]       : change classification from [m] to [n]  
-change_classification_from_to [m] [n]: change classification from [m] to [n]  
-change_extended_class_from_to [m] [n]: change extended class from [m] to [n]  
-change_extended_classification_from_to [m] [n]: change extended class from [m] to [n]  
-classify_attribute_above_as [m] [n] [o]: for attribute [m] with value above [n] set class to [o]  
-classify_attribute_below_as [m] [n] [o]: for attribute [m] with value below [n] set class to [o]  
-classify_attribute_between_as [m] [n] [o] [p]: for attribute [m] with value between [n] and [o] set class to [p]  
-classify_intensity_above_as [m] [n]: for intensity value above [m] set class to [n]  
-classify_intensity_below_as [m] [n]: for intensity value below [m] set class to [n]  
-classify_intensity_between_as [m] [n] [o]: for intensity value between [m] and [n] set class to [o]  
-classify_z_above_as [m] [n]        : for z value above [m] set class to [n]  
-classify_z_below_as [m] [n]        : for z value below [m] set class to [n]  
-classify_z_between_as [m] [n] [o]  : for z value between [m] and [n] set class to [o]  
-copy_classification_into_point_source: copy classification to point source  
-copy_classification_into_user_data : copy classification to user data  
-copy_intensity_into_classification : copy intensity to classification  
-copy_user_data_into_classification : copy user data into classification  
-drop_class [m] [n] [o]...          : drop points with class in [m][n][o]...  
-drop_classification [m] [n] [o]... : drop points with class in [m][n][o]...  
-drop_classification_mask [n]       : drop points with classification mask matches [n]  
-drop_extended_class [m] [n]...     : drop extended class [m] [n]...  
-drop_extended_classification [n]   : drop points with extended classification [n]  
-drop_extended_classification_mask [a] [b] [c] [d] [e] [f] [g] [h]: drop points with extended classification mask matches [a] [b] [c] [d] [e] [f] [g] [h]  
-keep_class [m] [n] [o]...          : keep points with class in [m][n][o]...  
-keep_classification [m] [n] [o]... : keep points with class in [m][n][o]...  
-keep_classification_mask [n]       : keep points with classification mask matches [n]  
-keep_extended_class [m] [n]...     : keep extended class [m] [n]...  
-keep_extended_classification [n]   : keep points with extended class [n]  
-move_ancient_to_extended_classification: move old data to extended classification  
-set_RGB_of_class [c] [r] [g] [b]   : set RGB values of class [c] to [r][g][b] (8 or 16 bit)  
-set_classification [n]             : set classification to [n]  
-set_extended_classification [n]    : set extended classification to [n]  

### Extra byte
-add_attribute_to_z [n]             : add value of attribute [n] to z value  
-add_scaled_attribute_to_user_data [m] [n]: scale attribute [m] value by [n] and add to user data  
-add_scaled_attribute_to_z [m] [n]  : scale attribute [m] value by [n] and add to z value  
-classify_attribute_above_as [m] [n] [o]: for attribute [m] with value above [n] set class to [o]  
-classify_attribute_below_as [m] [n] [o]: for attribute [m] with value below [n] set class to [o]  
-classify_attribute_between_as [m] [n] [o] [p]: for attribute [m] with value between [n] and [o] set class to [p]  
-copy_attribute_into_B [n]          : copy attribute [n] value into blue  
-copy_attribute_into_G [n]          : copy attribute [n] value into green  
-copy_attribute_into_I [n]          : copy attribute [n] value into intensity  
-copy_attribute_into_NIR [n]        : copy attribute [n] value into NIR (NearInfraRed)  
-copy_attribute_into_R [n]          : copy attribute [n] value into red  
-copy_attribute_into_intensity [n]  : copy attribute [n] value into intensity  
-copy_attribute_into_point_source [n]: copy attribute [n] value into point source  
-copy_attribute_into_register [m] [n]: copy attribute [m] value into register [m]  
-copy_attribute_into_user_data [n]  : copy attribute [n] value into user data field  
-copy_attribute_into_x [n]          : copy attribute [n] value into x  
-copy_attribute_into_y [n]          : copy attribute [n] value into y  
-copy_intensity_into_attribute [n]  : copy intensity to attribute [n] value  
-copy_register_into_attribute [m] [n]: copy register [m] to attribute [n] value  
-copy_user_data_into_attribute [n]  : copy user data into attribute [n] value  
-copy_z_into_attribute [n]          : copy z value into attribute [n] value  
-drop_attribute_above [m] [n]       : drop points with attribute [m] value > [n]  
-drop_attribute_below [m] [n]       : drop points with attribute [m] value < [n]  
-drop_attribute_between [m] [n] [o] : drop points with attribute [m] in range [n]...[o]  
-iadd_attribute [m] [n] [o] [p] [q] [r] [s] [t]: adds a new "extra_byte" attribute of data_type [m] name [n] description [o]; optional: scale[p] offset [q] pre_scale [r] pre_offset [s] no_data_value [t]  
-iadd_extra [m] [n] [o] [p] [q] [r] [s] [t]: adds a new "extra_byte" attribute of data_type [m] name [n] description [o]; optional: scale[p] offset [q] pre_scale [r] pre_offset [s] no_data_value [t]  
-keep_attribute_above [m] [n]       : keep points with attribute [m] value > [n]  
-keep_attribute_below [m] [n]       : keep points with attribute [m] value < [n]  
-keep_attribute_between [m] [n] [o] : keep points with attribute [m] in range [n]...[o]  
-load_attribute_from_text [m] [fnt] : load attribute [m] from file [fnt]  
-map_attribute_into_RGB [a] [fnm]   : map attribute [a] by table in file [fnm] to RGB values  
-scale_attribute [m] [n]            : scale attribute [m] by [n]  
-set_attribute [m] [n]              : set attribute [m] with value [n]  
-translate_attribute [m] [n]        : translate attribute [n] by [n]  

### Flags
-drop_keypoint                   : drop points flaged as keypoint  
-drop_overlap                    : drop points flaged as overlap  
-drop_scan_direction [n]         : drop points with scan direction [n]  
-drop_synthetic                  : drop points flaged as synthetic  
-drop_withheld                   : drop points flaged as withheld  
-keep_edge_of_flight_line        : keep points with "Edge of Flight Line" flag set  
-keep_keypoint                   : keep points flaged as keypoint  
-keep_overlap                    : keep points flaged as overlap  
-keep_scan_direction_change      : keep points with changed scan direction flag  
-keep_synthetic                  : keep points flaged as synthetic  
-keep_withheld                   : keep points flaged as withheld  
-set_edge_of_flight_line [0/1]   : set "Edge of Flight Line" flag to [0/1]  
-set_extended_overlap_flag [0/1] : set extended overlap flag to [0/1]  
-set_keypoint_flag [0/1]         : set keypoint flag to [0/1]  
-set_overlap_flag [0/1]          : set overlap flag to [0/1]  
-set_scan_direction_flag [0/1]   : set scan direction flag to [0/1]  
-set_synthetic_flag [0/1]        : set synthetic flag to [0/1]  
-set_withheld_flag [0/1]         : set withheld flag to [0/1]  

### GPS time
-adjusted_to_week                   : converts time stamps from Adjusted Standard GPS to GPS week  
-bin_gps_time_into_intensity [n]    : set intensity time to gps/[n]  
-bin_gps_time_into_point_source [n] : set point source to gps/[n]  
-drop_gps_time_above [n]            : drop points with GPS time above [n]  
-drop_gps_time_below [n]            : drop points with GPS time below [n]  
-drop_gps_time_between [m] [n]      : drop points with GPS time between [m] and [n]  
-drop_gpstime_above [n]             : drop points with GPS time above [n]  
-drop_gpstime_below [n]             : drop points with GPS time below [n]  
-drop_gpstime_between [m] [n]       : drop points with GPS time between [m] and [n]  
-keep_gps_time [m] [n]              : keep points with GPS time between [m] and [n]  
-keep_gps_time_above [n]            : keep points with GPS time above [n]  
-keep_gps_time_below [n]            : keep points with GPS time below [n]  
-keep_gps_time_between [m] [n]      : keep points with GPS time between [m] and [n]  
-keep_gpstime [m] [n]               : keep points with GPS time between [m] and [n]  
-keep_gpstime_above [n]             : keep points with GPS time above [n]  
-keep_gpstime_below [n]             : keep points with GPS time below [n]  
-keep_gpstime_between [m] [n]       : keep points with GPS time between [m] and [n]  
-set_gps_time [n]                   : set gps time to [n]  
-translate_gps_time [n]             : translate GPS time by [n]  

### Intensity
-bin_gps_time_into_intensity [n]    : set intensity time to gps/[n]  
-clamp_intensity [min] [max]        : limit intensity values to [min] and [max]  
-clamp_intensity_above [max]        : limit intensity values to maximal [max]  
-clamp_intensity_below [max]        : limit intensity values to minimal [min]  
-classify_intensity_above_as [m] [n]: for intensity value above [m] set class to [n]  
-classify_intensity_below_as [m] [n]: for intensity value below [m] set class to [n]  
-classify_intensity_between_as [m] [n] [o]: for intensity value between [m] and [n] set class to [o]  
-copy_B_into_intensity              : copy blue color value to intensity  
-copy_G_into_intensity              : copy green color value to intensity  
-copy_NIR_into_intensity            : copy NIR into intensity  
-copy_RGB_into_intensity            : copy weighted RGB value to intensity  
-copy_R_into_intensity              : copy red color value to intensity  
-copy_attribute_into_I [n]          : copy attribute [n] value into intensity  
-copy_attribute_into_intensity [n]  : copy attribute [n] value into intensity  
-copy_intensity_into_NIR            : copy intensity into NIR (NearInfraRed) value  
-copy_intensity_into_attribute [n]  : copy intensity to attribute [n] value  
-copy_intensity_into_classification : copy intensity to classification  
-copy_intensity_into_register [n]   : copy color intensitiy value into register [n]  
-copy_intensity_into_z              : copy intensity to z value  
-copy_register_into_intensity [n]   : copy register [n] into point intensitiy value  
-drop_intensity_above [n]           : drop points with intensity value above [n]  
-drop_intensity_below [n]           : drop points with intensity value below [n]  
-drop_intensity_between [m] [n]     : drop points with intensity value between [m] and [n]  
-iscale_intensity [n]               : scale intensity value by [n]  
-itranslate_intensity [n]           : translate input intensity by [n]  
-keep_NDVI_intensity_is_NIR [min] [max]: keep NDVI (Normalized Difference Vegetation Index) where intensity is NIR between [min] [max]  
-keep_intensity [m] [n]             : keep points with intensity between [m] and [n]  
-keep_intensity_above [n]           : keep points with intensity value above [n]  
-keep_intensity_below [n]           : keep points with intensity value below [n]  
-map_intensity [fnm]                : set the intensity by map in file [fnm]  
-scale_intensity [n]                : multiply intensity by [n]  
-set_intensity [n]                  : set intensity to [n]  
-switch_RGB_intensity_into_CIR      : set R to intensity; G to R; B to G  
-translate_intensity [n]            : translate intensity by [n]  
-translate_then_scale_intensity [m] [n]: translate intensity by [m] and scale by [n]  

### Raw point values
-clamp_raw_z [min] [max]            : limit raw z values to [min] and [max]  
-translate_raw_x [n]                : translate raw x value by [n]  
-translate_raw_xy_at_random [x] [y] : translate raw xy values by random and max offset of [x] [y]  
-translate_raw_xyz [x] [y] [z]      : translate raw coordinates by [x] [y] [z]  
-translate_raw_y [n]                : translate raw y value by [n]  
-translate_raw_z [n]                : translate raw z value by [n]  

### Registers
-add_registers [m] [n] [o]          : add register [m] and [n] and store result in register [o]  
-copy_B_into_register [n]           : copy blue color value into register [n]  
-copy_G_into_register [n]           : copy green color value into register [n]  
-copy_NIR_into_register [n]         : copy NearInfraRed value into register [n]  
-copy_R_into_register [n]           : copy red color value into register [n]  
-copy_attribute_into_register [m] [n]: copy attribute [m] value into register [m]  
-copy_intensity_into_register [n]   : copy color intensitiy value into register [n]  
-copy_point_source_into_register [n]: copy point source into register [n]  
-copy_register_into_B [n]           : copy register [n] into blue color value  
-copy_register_into_G [n]           : copy register [n] into green color value  
-copy_register_into_I [n]           : copy register [n] into NearInfraRed value  
-copy_register_into_NIR [n]         : copy register [n] into NearInfraRed value  
-copy_register_into_R [n]           : copy register [n] into red color value  
-copy_register_into_attribute [m] [n]: copy register [m] to attribute [n] value  
-copy_register_into_intensity [n]   : copy register [n] into point intensitiy value  
-copy_register_into_point_source [n]: copy register [n] to point source  
-copy_register_into_user_data [n]   : copy register [n] to user data  
-copy_register_into_x [n]           : copy register [n] to x value  
-copy_register_into_y [n]           : copy register [n] to y value  
-copy_register_into_z [n]           : copy register [n] to z value  
-copy_user_data_into_register [n]   : copy user data to register [n]  
-divide_registers [m] [n] [o]       : divide register [m] by register [n] and store result in register [o]  
-multiply_registers [m] [n] [o]     : Multiply register [m] with [n] and store result in register [o]  
-scale_register [m] [n]             : scale register index [m] with factor [n]  
-set_register [m] [n]               : set register [m] with value [n]  
-subtract_registers [m] [n] [o]     : subtract register [m] by register [n] and store result in register [o]  
-translate_register [m] [n]         : translate register index [m] value by [n]  

### Scan angle
-bin_abs_scan_angle_into_point_source [n]: set point source to scan_angle/[n]  
-drop_abs_scan_angle_above [max]    : drop points with absolute scan angle above [max]  
-drop_abs_scan_angle_below [min]    : drop points with absolute scan angle below [min]  
-drop_scan_angle_above [n]          : drop points with scan angle above [n]  
-drop_scan_angle_below [n]          : drop points with scan angle below [n]  
-drop_scan_angle_between [m] [n]    : drop points with scan angle between [m] and [n]  
-iscale_scan_angle [n]              : scale scan angle by [n]  
-itranslate_scan_angle [n]          : translate input scan angle by [n]  
-keep_scan_angle [m] [n]            : keep points with scan angle between [m] and [n]  
-keep_scan_angle_between [m] [n]    : keep points with scan angle between [m] and [n]  
-scale_scan_angle [n]               : scale scan angle by [n]  
-set_scan_angle [n]                 : set scan angle to [n]  
-translate_scan_angle [n]           : translate scan angle by [n]  
-translate_then_scale_scan_angle [m] [n]: translate scan angle by [m] and scale by [n]  

### Tiles
-keep_tile [x] [y] [size] : keep tile at lower-left [x] [y] with size [s]  
-tiles_ns [m] [n]         : create a tiling of DEMs with name [m] with tiles of size [n]  
-tiling_ns crater 500     : create a tiling of DEMs named 'crater' with tiles of size 500  

### Waveform packet
-drop_wavepacket [n]     : drop points with wavepacket value of [n]  
-flip_waveform_direction : flip the waveform direction in the waveform VLR  
-keep_wavepacket [n]     : keep points with wavepacket value of [n]  

### CRS
-aeac [m] [n] [meter/survey_feet/feet] [o] [p] [q] [r]: Albers Equal Area Conic Projection: False Easting [m] False Northing[n] [meter/survey_feet/feet] Central Meridian [o] Standard Parallel 1 [p] Standard Parallel 2 [q] Latitude of origin [r]  
-ecef                               : input is geocentric (Earth-centered Earth-fixed)  
-elevation_meter                    : use meter for elevation  
-elevation_survey_feet              : set vertical units from meters to US survey feet  
-elevation_surveyfeet               : use survey feet for elevation  
-ellipsoid [n]                      : use the WGS-84 ellipsoid [n]{do -ellipsoid -1 for a list of ellipsoids}  
-epsg [n]                           : set datum to EPSG [n]  
-etrs89                             : use datum ETRS89  
-gda2020                            : use datum GDA2020  
-gda94                              : use datum GDA94  
-grs80                              : use datum GRS1980  
-latlong                            : geometric coordinates in latitude/longitude order  
-lcc 609601.22 0.0 meter 33.75 -79 34.33333 36.16666: specifies a lambertian conic confomal projection  
-longlat                            : geometric coordinates in longitude/latitude order  
-meter                              : use meter  
-nad27                              : use the NAD27 ellipsoid  
-nad83                              : use the NAD83 ellipsoid  
-nad83_2011                         : use datum NAD83_2011  
-nad83_csrs                         : use datum NAD83_CSRS  
-nad83_harn                         : use datum NAD83_HARN  
-nad83_pa11                         : set horizontal datum to NAD83 PA11  
-osgb1936                           : use datum OSGB 1936  
-sp27 SC_N                          : use the NAD27 South Carolina North state plane  
-sp83 CO_S                          : use the NAD83 Colorado South state plane for georeferencing  
-survey_feet                        : use survey feet  
-surveyfeet                         : use survey feet as unit of measurement  
-target_aeac [m] [n] [meter/survey_feet/feet] [o] [p] [q] [r]: Albers Equal Area Conic Projection for target: False Easting [m] False Northing[n] [meter/survey_feet/feet] Central Meridian [o] Standard Parallel 1 [p] Standard Parallel 2 [q] Latitude of origin [r]  
-target_ecef                        : output is geocentric (Earth-centered Earth-fixed)  
-target_elevation_feet              : output uses feet for elevation  
-target_elevation_meter             : output uses meter for elevation  
-target_elevation_precision [n]     : output uses [n] (meter/feet) resolution for z  
-target_elevation_survey_feet       : use elevation survey feet as target unit  
-target_elevation_surveyfeet        : output uses survey feet for elevation  
-target_epsg [n]                    : output is EPSG code [n] (e.g. 2193=NZGD2000)  
-target_feet                        : output uses feet  
-target_latlong                     : output is geometric coordinates in latitude/longitude  
-target_lcc 609601.22 0.0 meter 33.75 -79 34.33333 36.16666: specifies a lambertian conic confomal projection at target  
-target_longlat                     : output is geometric coordinates in longitude/latitude  
-target_meter                       : output uses meter  
-target_precision [n]               : output uses [n] (meter/feet) resolution for x and y  
-target_sp27 SC_N                   : output is state plane NAD27 South Carolina North  
-target_sp83 CO_S                   : output is state plane NAD83 Colorado South  
-target_survey_feet                 : output uses survey feet  
-target_surveyfeet                  : use survey feet as target unit  
-target_tm                          : use transverse mercator projection for target  
-target_utm 12T                     : output is UTM zone 12T  
-tm 609601.22 0.0 meter 33.75 -79 0.99996: specifies a transverse mercator projection  
-transverse_mercator                : use transverse mercator projection  
-utm 12T                            : use UTM zone 12T  
-vertical_cgvd2013                  : set vertical datum to CGVD2013  
-vertical_cgvd28                    : set vertical datum to CGVD28  
-vertical_dhhn2016                  : set vertical datum to DHHN2016  
-vertical_dhhn92                    : set vertical datum to DHHN92  
-vertical_dvr90                     : set vertical datum to DVR90  
-vertical_epsg [n]                  : set vertical datum to EPSG [n]  
-vertical_evrf2007                  : set vertical datum to EVRF2007  
-vertical_navd29                    : set vertical datum to NAVD29  
-vertical_navd88                    : set vertical datum to NAVD88  
-vertical_ngvd29                    : set vertical datum to NGVD29  
-vertical_nn2000                    : set vertical datum to NN2000  
-vertical_nn54                      : set vertical datum to NN54  
-vertical_nzvd2016                  : set vertical datum to NZVD2016  
-vertical_wgs84                     : set vertical datum to WGS84  
-wgs72                              : use the WGS-72 ellipsoid  
-wgs84                              : use the WGS-84 ellipsoid  

### Logical
-filter_and         : boolean AND combination of last 2 filters  
-filter_or          : boolean OR combination of last 2 filters  
-filtered_transform : do the transformation only on points of the current filter  

### Input
-i [fnp]        : input file or input file mask [fnp] (e.g. *.laz;fo?.la?;esri.shp,...)  
-io_ibuffer [n] : use read-input-buffer of size [n] bytes  
-iparse [xyz]   : define fields [xyz] for text input parser  
-ipts           : input as PTS (plain text lidar source), store header in VLR  
-iptx           : input as PTX (plain text extended lidar data), store header in VLR  
-iptx_transform : use PTX file header to transform point data  
-iskip [n]      : skip [n] lines at the beginning of the text input  
-itxt           : expect input as text file  
-lof [fnf]      : use input out of a list of files [fnf]  
-unique         : remove duplicate files in a -lof list  
-merged         : merge input files  
-stdin          : pipe from stdin  

### Output
-compression_quality [n] : set compression quality to [n] for compressable image formats  
-do_not_populate         : do not populate header on output  
-nil                     : pipe output to NULL (suppress output)  
-o [n]                   : use [n] as output file  
-oasc                    : output as ascii file  
-obil                    : output as bil (Band Interleaved by Line)  
-ocsv                    : output as CSV (comma separated value)  
-ocut [n]                : cut the last [n] characters from name  
-odir [n]                : set output directory to [n]  
-odix [n]                : set output file name suffix to [n]  
-odtm                    : output as dtm (Digital Terrain Models)  
-oflt                    : output as flt (Float file format)  
-oforce                  : force output creation also on errors or warnings  
-oimg                    : output as img (Image file) (Win32 only)  
-ojpg                    : output as jpg (JPG image)  
-olaz                    : output as LAZ (compressed LAS)  
-opng                    : output as png (PNG image)  
-osep [sep]              : set text output separator as [sep](see table below)  
-otif                    : output as GeoTIFF image  
-oxyz                    : output as xyz textfile  
-force_tif               : force output in TIFF format, regardless of data size. Used in combination with GeoTIFF output  
-force_bigTif            : force output in BigTIFF format, regardless of data size. Used in combination with GeoTIFF output  
-pipe_on                 : write output to command pipe, see also -std_in  
-populate                : populate header on output  
-target_ecef             : output is geocentric (Earth-centered Earth-fixed)  
-temp_files [n]          : set base file name [n] for temp files (example: E:\tmp)

### parse
The '-parse [xyz]' flag specifies how to interpret each line of the ASCII file.
For example, 'tsxyzssa' means that the first number is the gpstime, the next
number should be skipped, the next three numbers are the x, y, and z coordinate,
the next two should be skipped, and the next number is the scan angle.

The other supported entries are:  
  x : [x] coordinate  
  y : [y] coordinate  
  z : [z] coordinate  
  t : gps [t]ime  
  R : RGB [R]ed channel  
  G : RGB [G]reen channel  
  B : RGB [B]lue channel  
  I : N[I]R channel of LAS 1.4 point type 8  
  s : [s]kip a string or a number that we don't care about  
  i : [i]ntensity  
  a : scan [a]ngle  
  n : [n]umber of returns of that given pulse  
  r : number of [r]eturn  
  h : with[h]eld flag  
  k : [k]eypoint flag  
  g : synthetic fla[g]  
  o : [o]verlap flag of LAS 1.4 point types 6, 7, 8  
  l : scanner channe[l] of LAS 1.4 point types 6, 7, 8  
  E : terrasolid [E]hco Encoding  
  c : [c]lassification. If extended classes are used: Use o,l or I to force 1.4 format.  
  u : [u]ser data  
  p : [p]oint source ID  
  e : [e]dge of flight line flag  
  d : [d]irection of scan flag  
  0-9 : additional attributes described as extra bytes (0 through 9)  
  (13) : additional attributes described as extra bytes (10 and up)  
  H : a hexadecimal string encoding the RGB color  
  J : a hexadecimal string encoding the intensity  

### output separator
The '-osep [sep]' argument specifies the output format of a text(xyz or csv) output.
Supported [sep] values:

  comma
  tab
  dot
  colon
  semicolon
  hyphen
  space
  
## Licensing

Info on licensing and pricing: https://rapidlasso.de/pricing/.
If you have any questions or need assistance, email to info@rapidlasso.de.

## Evaluation and demo mode

Please use the "-demo" argument to run the tool in demo mode. For quality tests,
use small files (< 1.5 million points). If you use larger files, the output will
contain diagonal lines/output distortions due to the license protection.

## Support

1. We invite you to join our LAStools Google Group (http://groups.google.com/group/lastools/).
   If you are looking for information about a specific tool, enter the tool name in the search 
   function and you'll find all discussions related to the respective tool. 
2. Customer Support Page: https://rapidlasso.de/customer-support/.  
3. Download LAStools: https://rapidlasso.de/downloads/.  
4. Changelog: https://rapidlasso.de/changelog/.  


If you want to send us feedback or have questions that are not answered in the resources above, 
please email to info@rapidlasso.de.
